<?php
declare(strict_types=1);

require __DIR__ . '/../app/init.php';

use App\DB;
use App\Env;
use App\Response;

$db = DB::pdo();

function tableExists(PDO $pdo, string $table): bool {
    $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
    $stmt->execute([$table]);
    return (bool) $stmt->fetchColumn();
}

$usersTable   = 'users';
$modulesTable = 'modules';
$fieldsTable  = 'fields';

$installed = tableExists($db, $usersTable) && tableExists($db, $modulesTable) && tableExists($db, $fieldsTable);

$errors = [];
$done = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $adminUser = trim((string)($_POST['admin_user'] ?? 'admin'));
    $adminPass = (string)($_POST['admin_pass'] ?? 'admin123');

    if ($adminUser === '' || strlen($adminUser) < 3) $errors[] = "Usuário admin inválido (mín. 3 caracteres).";
    if ($adminPass === '' || strlen($adminPass) < 6) $errors[] = "Senha admin inválida (mín. 6 caracteres).";

    if (!$errors) {
        $db->exec("CREATE TABLE IF NOT EXISTS `users` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `username` VARCHAR(100) NOT NULL UNIQUE,
            `password_hash` VARCHAR(255) NOT NULL,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

        $db->exec("CREATE TABLE IF NOT EXISTS `modules` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `name` VARCHAR(150) NOT NULL,
            `table_name` VARCHAR(64) NOT NULL UNIQUE,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

        $db->exec("CREATE TABLE IF NOT EXISTS `fields` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `module_id` INT UNSIGNED NOT NULL,
            `name` VARCHAR(64) NOT NULL,
            `label` VARCHAR(150) NOT NULL,
            `type` VARCHAR(32) NOT NULL,
            `required` TINYINT(1) NOT NULL DEFAULT 0,
            `options` TEXT NULL,
            `position` INT NOT NULL DEFAULT 0,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `uniq_module_field` (`module_id`,`name`),
            CONSTRAINT `fk_fields_module` FOREIGN KEY (`module_id`) REFERENCES `modules`(`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

        // Create admin if not exists
        $stmt = $db->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$adminUser]);
        $exists = $stmt->fetchColumn();

        if (!$exists) {
            $hash = password_hash($adminPass, PASSWORD_DEFAULT);
            $ins = $db->prepare("INSERT INTO users (username, password_hash) VALUES (?, ?)");
            $ins->execute([$adminUser, $hash]);
        }

        $done = true;
    }
}

$title = "Instalação - PHP CRUD Builder";
?>
<!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?= htmlspecialchars($title, ENT_QUOTES, 'UTF-8') ?></title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
</head>
<body class="bg-light">
<div class="container py-5">
  <div class="row justify-content-center">
    <div class="col-lg-7">
      <div class="card shadow-sm">
        <div class="card-body p-4">
          <h1 class="h4 mb-3">Instalação</h1>

          <?php if ($done): ?>
            <div class="alert alert-success">
              Instalado com sucesso. Você já pode acessar o painel.
            </div>
            <a class="btn btn-primary" href="index.php?a=login">Ir para o Login</a>
          <?php else: ?>

            <?php if ($installed): ?>
              <div class="alert alert-warning">
                Parece que as tabelas já existem. Se você quer reinstalar, apague o banco/tabelas antes.
              </div>
              <a class="btn btn-outline-primary" href="index.php?a=login">Ir para o Login</a>
            <?php endif; ?>

            <?php if ($errors): ?>
              <div class="alert alert-danger">
                <ul class="mb-0">
                  <?php foreach ($errors as $e): ?>
                    <li><?= htmlspecialchars($e, ENT_QUOTES, 'UTF-8') ?></li>
                  <?php endforeach; ?>
                </ul>
              </div>
            <?php endif; ?>

            <p class="text-muted">Este instalador cria as tabelas do sistema e o usuário administrador.</p>

            <form method="post" class="mt-3">
              <div class="mb-3">
                <label class="form-label">Usuário admin</label>
                <input class="form-control" name="admin_user" value="admin" required>
              </div>
              <div class="mb-3">
                <label class="form-label">Senha admin</label>
                <input class="form-control" name="admin_pass" type="password" value="admin123" required>
              </div>
              <button class="btn btn-primary">Instalar</button>
            </form>

          <?php endif; ?>

          <hr class="my-4">
          <div class="small text-muted">
            Dica: configure o arquivo <code>.env</code> antes de instalar.
          </div>

        </div>
      </div>
    </div>
  </div>
</div>
</body>
</html>
